import ChecksumService from '../utils/ChecksumService';
import AlreadyUploadedException from "../exceptions/AlreadyUploadedException";


const CHUNK_SIZE = 1024 * 1024;

class ChunksUploader {

    constructor(file, endpoint, api_key, checksum = null, chunk_size = CHUNK_SIZE) {
        this.file = file;
        this.endpoint = endpoint;
        this.api_key = api_key;
        this.chunk_size = chunk_size;
        this.chunk_statuses = [];
        this.nb_chunks = Math.ceil(this.file.size / this.chunk_size);
        for (let i = 0; i < this.nb_chunks; i++) {
            this.chunk_statuses[i] = {n: i, done: false};
        }
        this.chunk_indexes = [];
        let byteIndex = 0
        for (let i = 0; i < this.nb_chunks; i++) {
            let byteEnd = Math.ceil((this.file.size / this.nb_chunks) * (i + 1));
            this.chunk_indexes.push({start: byteIndex, end: byteEnd})
            byteIndex += (byteEnd - byteIndex);
        }
        this.checksum = checksum;
    }


     fixedEncodeURIComponent (str) {
        return encodeURIComponent(str).replace(/[!'()*]/g, function(c) {
            return '%' + c.charCodeAt(0).toString(16);
        });
    }

    /**
     * @param {Integer} chunk_num - chunk number
     * @param {String} path - collection path
     * @return {Promise} Promise object of XHR
     */
    uploadChunk(chunk_num, path) {
        let data = this.file.slice(this.chunk_indexes[chunk_num].start, this.chunk_indexes[chunk_num].end);
        let options =
            {
                url: this.endpoint + 'upload/partial/',
                method: 'POST',
                headers: {
                    "X-Api-Key": this.api_key,
                    "X-file-chunk": (chunk_num + 1) + "/" + this.nb_chunks,
                    "X-file-hash": this.checksum,
                    "X-file-name": window.btoa(unescape(this.fixedEncodeURIComponent( this.file.name )))
                }
            }
        if (path)
            options.headers['X-origin-dir'] = path


        return new Promise((resolve, reject) => {
            let xhr = new XMLHttpRequest();

            xhr.open(options.method, options.url);

            xhr.onload = () => {
                if (xhr.status >= 200 && xhr.status < 300) {
                    this.chunk_statuses[chunk_num].done = true;
                    resolve(xhr);
                } else {
                    this.chunk_statuses[chunk_num].done = false;
                    reject({
                        status: xhr.status,
                        statusText: xhr.statusText
                    });
                }
            };

            xhr.onerror = (e) => {
                console.log(e);
                reject({
                    status: xhr.status,
                    statusText: xhr.statusText
                });
            };

            // Set headers
            if (options.headers) {
                Object.keys(options.headers).forEach((key) => {
                    xhr.setRequestHeader(key, options.headers[key]);
                });
            }

            xhr.send(data);
        });
    }


    setChecksum(cs) {
        this.checksum = cs;
    }


    getNbChunks() {
        return this.nb_chunks;
    }

}



const upload_events = {
    start_upload_event : 'start_upload',
    end_upload_event: 'end_upload',
    start_file_upload_event : 'start_file_upload',
    end_file_upload_event : 'end_file_upload',
    resume_upload_event : 'resume_upload',
    chunk_uploaded_event: 'chunk_uploaded'
}
const ACCEPTED_EXTENSIONS = 'pdf|ai|bmp|tif|tiff|jpg|jpeg|png';


class JamaClient {

    constructor(endpoint, api_key) {
        this.endpoint = endpoint;
        this.api_key = api_key;
        this.id = 0;
        this.checksumService = new ChecksumService();
        this.listeners = [];
    }

    uploadChunk(chunkNumber, uploader, path){
        return new Promise((resolve, reject) => {
            uploader.uploadChunk(chunkNumber, path).then((r) => {
                resolve(r);
            }).catch((e) => {
                console.log(e);
                reject({error: e, chunk: chunkNumber})
            })
        })
    }

    async uploadFile(file, path) {
        let uploader = new ChunksUploader(file, this.endpoint, this.api_key);
        let cs = await this.checksumService.sha256(file);
        this.dispatchEvent(new CustomEvent(upload_events.start_file_upload_event, { 'detail': {
                hash: cs,
                name: file.name,
                path: path,
                nb_chunks: uploader.getNbChunks()
            } })
        );
        uploader.setChecksum(cs);

        //checks if already uploaded
        let uploadInfos = await this.uploadInfos(cs);
        if(uploadInfos.status === 'available') {
            throw new AlreadyUploadedException(uploadInfos.id)//'File Already Uploaded : ' + file.name;
        }
        else {
            let errors = [];
            for (let i = 0; i < uploader.getNbChunks(); i++) {
                try {
                    let chunkResult = await this.uploadChunk(i, uploader, path);
                    this.dispatchEvent(new CustomEvent(upload_events.chunk_uploaded_event, { 'detail': {
                            hash: cs,
                            name: file.name,
                            chunk : i,
                            result : chunkResult
                        } })
                    );
                } catch (err) {
                    console.log(err);
                    errors.push(err)
                }
            }

            // await Promise.all(promises).catch((e) => {
            //     console.log('error slice ' + e.chunk);
            //     console.log(e.error)
            // })


            let statusInfos = await this.uploadInfos(cs);
            //console.log('will dispatch end resource event');
            //console.log(statusInfos)
            this.dispatchEvent(new CustomEvent(upload_events.end_file_upload_event, {
                    'detail': {
                        hash: cs,
                        status: statusInfos.status,
                        errors: errors
                    }
                })
            );

            return {
                resource: file.name,
                id: statusInfos.id,
                status: statusInfos.status,
                available_chunks: statusInfos.available_chunks
            }
        }

    }

    /**
     *
     * @param files
     * @param collectionPath
     * @returns {Promise<Array>}
     */
    async uploadFiles(files, collectionPath = null){
        this.dispatchEvent(new CustomEvent(upload_events.start_upload_event,{detail: {nb_files : files.length}}));
        //let promises = [];
        let errors = [];
        let uploadResults = [];
        for(let i = 0 ; i < files.length; i++){//sequential upload promises to manage errors
            if(this.checkExtension(files[i])) {
                let path = (files[i]).webkitRelativePath.replace((files[i]).name,'')
                if(collectionPath)
                    path = collectionPath +'/' + path;
                //promises.push(this.uploadFile(resources[i], path));

                try {
                    //console.log('will upload file ...')
                    let uploadResult = await this.uploadFile(files[i], path);
                    console.log(uploadResult)
                    uploadResults.push(uploadResult);
                }
                catch(e){
                    console.log(e);
                    errors.push(e);
                }

            }
            else{
                errors.push((files[i]).name + ' : Format de fichier incorrect.\n\n' +
                    'Veuillez choisir un format de fichier parmi :\n\n'
                    + ACCEPTED_EXTENSIONS.replaceAll('|', ', ') + ' .');
            }
        }

        // //sequential promises to manage errors
        // let uploadResults = []
        // for(let i = 0 ; i < promises.length; i++){
        //     try {
        //         console.log();
        //         let uploadResult = await (promises[i]);
        //         uploadResults.push(uploadResult);
        //     }
        //     catch(e){
        //         console.log(e);
        //         errors.push(e);
        //     }
        // }

        // for(let i = 0; i < uploadResults.length; i++) {
        //     let uploadResult = uploadResults[i];
        //     if (uploadResult.status !== 'available') {
        //         console.log('erreur du le fichier ', uploadResult.resource)
        //         console.log(uploadResult.status)
        //     }
        // }
        //console.log('will dispatch end upload event')
        this.dispatchEvent(new CustomEvent(upload_events.end_upload_event,{detail : uploadResults}))
        return errors;
    }


    checkExtension(file) {
        let val = file.name.toLowerCase(),
            regex = new RegExp("(.*?)\\.(" + ACCEPTED_EXTENSIONS + ")$");
        return regex.test(val)

    }


    /**
     * @param file
     * @param uploadInfos
     * {
     *     hash : 1234,
     *     name: 'my_file_name',
     *     path: 'my_file_path/',
     *     nb_chunks: X,
     *     available_chunks: [...]
     * }
     */
    async resumeUpload(file, uploadInfos){

        this.dispatchEvent(new CustomEvent(upload_events.resume_upload_event),{detail : {hash : uploadInfos.hash}});
        const lpad = function (value, padding) {
            var zeroes = new Array(padding+1).join("0");
            return (zeroes + value).slice(-padding);
        }

        let errors = [];
        let uploader = new ChunksUploader(file, this.endpoint, this.api_key,  uploadInfos.hash);
        //uploads missing chunks
        for(let i = 0; i < uploadInfos.nb_chunks; i++){
            if(uploadInfos.available_chunks.indexOf('202-'+lpad(i+1,3)+'.part') < 0 ){
                try {
                    let chunkResult = await this.uploadChunk(i, uploader, uploadInfos.path);
                    this.dispatchEvent(new CustomEvent(upload_events.chunk_uploaded_event, { 'detail': {
                            hash: uploadInfos.hash,
                            name: uploadInfos.name,
                            chunk : i,
                            result : chunkResult
                        } })
                    );
                } catch (err) {
                    console.log(err);
                    errors.push(err)
                }

                //await this.uploadChunk(i, uploader, uploadInfos.path)
            }
        }



        // await Promise.all(promises).catch((e) => {
        //     console.log('error slice ' + e.chunk);
        //     //console.log(e.error)
        // })
        let statusInfos = await this.uploadInfos(uploadInfos.hash);

        this.dispatchEvent(new CustomEvent(upload_events.end_file_upload_event, { 'detail': {
                hash: uploadInfos.hash,
                status: statusInfos.status,
                errors : errors
            } })
        );


        this.dispatchEvent(new Event(upload_events.end_upload_event));

    }

    addListener(listener){
        if(this.listeners.indexOf(listener) < 0)
            this.listeners.push(listener);
    }

    dispatchEvent(event){
        //console.log('JAMACLIENT EMIT EVENT ', event.type)
        this.listeners.forEach( listener => listener.dispatchEvent(event))
    }


    getMediaAsBlob(mediaId){
        let headers = {
            "X-Api-Key": this.api_key
        }
        return fetch(this.endpoint +"download/"+mediaId, { headers }).then(response => response.blob())
    }

    getMediaURL(mediaId){
        return this.endpoint +"download/"+mediaId
    }

    /** RPC FUNCTIONS **/


    /**
     *
     * @param {String} method
     * @param {Array} params
     * @returns {Any}
     */
    async _rpc(method, params) {
        this.id++;
        let rpc_request = {
            id: this.id,
            method: method,
            params: params
        };
        let body = JSON.stringify(rpc_request);
        let headers = new Headers();
        headers.set('X-Api-Key', this.api_key);
        let response = await fetch(this.endpoint, {
            method: 'POST',
            headers: headers,
            body: body
        }).catch(() => {
            return false
        });
        if (response.ok) {
            let response_dict = await response.json();
            if (response_dict.error) {
                throw response_dict.error;
            }
            return response_dict.result;
        } else return false
    }

    /**
     * Add access to the RPC API for the given user name with the given API key.
     * A new user will be created if none is available with given user name.
     *
     * Note: only user with admin privileges can use this.
     *
     * @param {String} user_name
     * @param {String} api_key
     * @returns {Boolean}
     */
    async activateRpcAccess(user_name, api_key) {
        return this._rpc("activate_rpc_access", [user_name, api_key]);
    }


    /**
     * Create a new collection based on 'title'. If 'parent_id' is set,
     * will create new collection as child of parent. Otherwise, the new
     * collection is created at root.
     *
     * Returns either the serialized new collection of null if parent does
     * not exist.
     *
     * Example output:
     *
     * ```
     * {
     *     "id": 3,
     *     "title": "paintings",
     *     "resources_count": 0,
     *     "children_count": 0,
     *     "descendants_count": 0,
     *     "descendants_resources_count": 0,
     *     "parent": null,
     *     "children": null,
     *     "metas": [],
     *     "public_access": false,
     *     "tags": [],
     * }
     * ```
     *
     * @param {String} title
     * @param {BigInteger} parent_id
     * @returns {Object}
     */
    async addCollection(title, parent_id = null) {
        return this._rpc("add_collection", [title, parent_id]);
    }


    /**
     * Will take a path such as '/photos/arts/paintings/'
     * and build the corresponding hierarchy of collections. The hierarchy
     * is returned as a list of serialized collections.
     *
     * Beware: Because the collections are serialized before their children,
     * all the children/descendants counts are set to 0.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *         "id": 1,
     *         "title": "photos",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": null,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 2,
     *         "title": "arts",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 1,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 3,
     *         "title": "paintings",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 2,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     * ]
     * ```
     *
     * @param {String} path
     * @returns {Array}
     */
    async addCollectionFromPath(path) {
        return this._rpc("add_collection_from_path", [path]);
    }


    /**
     * Add a meta value to a collection given their ids.
     *
     * @param {BigInteger} collection_id
     * @param {BigInteger} meta_id
     * @param {String} meta_value
     * @returns {Any}
     */
    async addMetaToCollection(collection_id, meta_id, meta_value) {
        return this._rpc("add_meta_to_collection", [collection_id, meta_id, meta_value]);
    }


    /**
     * Add a meta value to a resource given their ids.
     *
     * @param {BigInteger} resource_id
     * @param {BigInteger} meta_id
     * @param {String} meta_value
     * @returns {Any}
     */
    async addMetaToResource(resource_id, meta_id, meta_value) {
        return this._rpc("add_meta_to_resource", [resource_id, meta_id, meta_value]);
    }


    /**
     * Add a new metadata to metadata set.
     *
     * Set optional 'metadata_type_id'
     *
     * @param {String} title
     * @param {BigInteger} metas_set_id
     * @param {BigInteger} metadata_type_id
     * @returns {Any}
     */
    async addMetadata(title, metas_set_id, metadata_type_id = null) {
        return this._rpc("add_metadata", [title, metas_set_id, metadata_type_id]);
    }


    /**
     * Create new metadata set from title.
     *
     * @param {String} title
     * @returns {BigInteger}
     */
    async addMetadataset(title) {
        return this._rpc("add_metadataset", [title]);
    }


    /**
     * Add a resource to a collection given ids.
     *
     * @param {BigInteger} resource_id
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async addResourceToCollection(resource_id, collection_id) {
        return this._rpc("add_resource_to_collection", [resource_id, collection_id]);
    }


    /**
     * Add tag to a collection based on tag uid and collection id.
     *
     * @param {String} tag_uid
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async addTagToCollection(tag_uid, collection_id) {
        return this._rpc("add_tag_to_collection", [tag_uid, collection_id]);
    }


    /**
     * Add tag to a resource based on tag uid and resource id.
     *
     * @param {String} tag_uid
     * @param {BigInteger} resource_id
     * @returns {Boolean}
     */
    async addTagToResource(tag_uid, resource_id) {
        return this._rpc("add_tag_to_resource", [tag_uid, resource_id]);
    }


    /**
     * Performs a complex search using terms such as 'contains', 'is', 'does_not_contain'.
     *
     * Multiple conditions can be added.
     *
     * Example input:
     *
     * ```
     * [
     *     {"property": "title", "term": "contains", "value": "cherbourg"},
     *     {"meta": 123, "term": "is", "value": "35mm"},
     *     {"tags": ["PAINTINGS", "PHOTOS"]}
     * ]
     * ```
     *
     * Example output:
     *
     * ```
     * {
     *     "collections": [],
     *     "resources": [
     *         {
     *         "id": 1,
     *         "title": "Cherbourg by night",
     *         "original_name": "cherbourg_by_night.jpg",
     *         "type": "image/jpeg",
     *         "hash": "0dd93a59aeaccfb6d35b1ff5a49bde1196aa90dfef02892f9aa2ef4087d8738e",
     *         "metas": null,
     *         "urls": [],
     *         "tags": [],
     *         }
     *     ]
     * }
     * ```
     *
     * @param {Array} search_terms
     * @returns {Object}
     */
    async advancedSearch(search_terms) {
        return this._rpc("advanced_search", [search_terms]);
    }


    /**
     * Return terms conditions to be used in advanced search.
     *
     * Example output:
     *
     * ```
     * [
     *     "is",
     *     "contains",
     *     "does_not_contain"
     * ]
     * ```
     *
     * @returns {Array}
     */
    async advancedSearchTerms() {
        return this._rpc("advanced_search_terms", []);
    }


    /**
     * Get ancestors from collection id as a list of serialized collections.
     *
     * If 'include_self' is true, will add the current collection at the begining.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *         "id": 1,
     *         "title": "photos",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": null,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 2,
     *         "title": "arts",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 1,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 3,
     *         "title": "paintings",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 2,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     * ]
     * ```
     *
     * @param {BigInteger} collection_id
     * @param {Boolean} include_self
     * @returns {Array}
     */
    async ancestorsFromCollection(collection_id, include_self = false) {
        return this._rpc("ancestors_from_collection", [collection_id, include_self]);
    }


    /**
     * Get ancestors from resource id as a list of serialized collections.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *         "id": 1,
     *         "title": "photos",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": null,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 2,
     *         "title": "arts",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 1,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     *     {
     *         "id": 3,
     *         "title": "paintings",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": 2,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *     },
     * ]
     * ```
     *
     * @param {BigInteger} resource_id
     * @returns {Array}
     */
    async ancestorsFromResource(resource_id) {
        return this._rpc("ancestors_from_resource", [resource_id]);
    }


    /**
     * Change the value of a meta for a collection.
     *
     * @param {BigInteger} meta_value_id
     * @param {String} meta_value
     * @returns {Boolean}
     */
    async changeCollectionMetaValue(meta_value_id, meta_value) {
        return this._rpc("change_collection_meta_value", [meta_value_id, meta_value]);
    }


    /**
     * Change the value of a meta for a resource
     *
     * @param {BigInteger} meta_value_id
     * @param {String} meta_value
     * @returns {Boolean}
     */
    async changeResourceMetaValue(meta_value_id, meta_value) {
        return this._rpc("change_resource_meta_value", [meta_value_id, meta_value]);
    }


    /**
     * Get a particular collection given its id.
     *
     * Example output:
     *
     * ```
     * {
     *     "id": 2,
     *     "title": "art works",
     *     "resources_count": 23,
     *     "children_count": 5,
     *     "descendants_count": 12,
     *     "descendants_resources_count": 58,
     *     "parent": 1,
     *     "children": None,
     *     "metas": [],
     *     "public_access": False,
     *     "tags": [],
     * }
     * ```
     *
     * if collection id is null, will only return the number of collections at root:
     *
     * ```
     * {
     *     "id": None,
     *     "title": "root",
     *     "children_count": 25,
     * }
     * ```
     *
     * @param {BigInteger} collection_id
     * @returns {Object}
     */
    async collection(collection_id = null) {
        return this._rpc("collection", [collection_id]);
    }


    /**
     * Return the user's collections under the parent collection
     * specified by 'parent_id'. If 'parent_id' is null, will return
     * collections available at root. If 'recursive' is true, will
     * return all the descendants recursively in the 'children' key.
     * If recursive is false, 'children' is null.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *         "id": 2,
     *         "title": "art works",
     *         "resources_count": 23,
     *         "children_count": 5,
     *         "descendants_count": 12,
     *         "descendants_resources_count": 58,
     *         "parent": 1,
     *         "children": None,
     *         "metas": [],
     *         "public_access": False,
     *         "tags": [],
     *     }
     * ]
     * ```
     *
     * @param {BigInteger} parent_id
     * @param {Boolean} recursive
     * @param {BigInteger} limit_from
     * @param {BigInteger} limit_to
     * @returns {Array}
     */
    async collections(parent_id = null, recursive = false, limit_from = 0, limit_to = 2000) {
        return this._rpc("collections", [parent_id, recursive, limit_from, limit_to]);
    }


    /**
     * Deactivate access to the RPC API for the given user name and API key.
     * Only the access (API key) is removed, not the user.
     *
     * Note: only user with admin privileges can use this.
     *
     * @param {String} user_name
     * @param {String} api_key
     * @returns {Boolean}
     */
    async deactivateRpcAccess(user_name, api_key) {
        return this._rpc("deactivate_rpc_access", [user_name, api_key]);
    }


    /**
     * Delete collection given its id.
     *
     * Collection MUST be empty of any content (no children collections and no resources),
     * unless the 'recursive'parameter is set to True.
     *
     * @param {BigInteger} collection_id
     * @param {Boolean} recursive
     * @returns {Object}
     */
    async deleteCollection(collection_id, recursive = false) {
        return this._rpc("delete_collection", [collection_id, recursive]);
    }


    /**
     * Delete metadata based on its id.
     *
     * @param {BigInteger} metadata_id
     * @returns {Boolean}
     */
    async deleteMetadata(metadata_id) {
        return this._rpc("delete_metadata", [metadata_id]);
    }


    /**
     * Delete metadata set based on its id. Optional recursive
     * call.
     *
     * @param {BigInteger} metadataset_id
     * @param {Boolean} recursive
     * @returns {Object}
     */
    async deleteMetadataset(metadataset_id, recursive = false) {
        return this._rpc("delete_metadataset", [metadataset_id, recursive]);
    }


    /**
     * Permanently delete a resource given its id.
     *
     * @param {BigInteger} resource_id
     * @returns {Boolean}
     */
    async deleteResource(resource_id) {
        return this._rpc("delete_resource", [resource_id]);
    }


    /**
     * Get one particular metadata given its id.
     *
     * Example output:
     *
     * ```
     * {
     *     "id": 2,
     *     "title": "ICC_Profile:GrayTRC",
     *     "set_id": 1,
     *     "set_title": "exif metas",
     *     "rank": 1,
     *     "owner": "john",
     * }
     * ```
     *
     * @param {BigInteger} metadata_id
     * @returns {Object}
     */
    async metadata(metadata_id) {
        return this._rpc("metadata", [metadata_id]);
    }


    /**
     * Get all metadatas given a metadata set id.
     *
     * Metadatas MAY be ordered with the rank attribute.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *         "id": 1,
     *         "title": "PNG:ProfileName",
     *         "set_id": 1,
     *         "set_title": "exif metas",
     *         "rank": 0,
     *         "owner": "john",
     *     },
     *     {
     *         "id": 2,
     *         "title": "ICC_Profile:GrayTRC",
     *         "set_id": 1,
     *         "set_title": "exif metas",
     *         "rank": 1,
     *         "owner": "john",
     *     }
     * ]
     * ```
     *
     * @param {BigInteger} metadata_set_id
     * @returns {Array}
     */
    async metadatas(metadata_set_id) {
        return this._rpc("metadatas", [metadata_set_id]);
    }


    /**
     * Get the list of all the user's metadata sets.
     * For each metadatas set, the number of metadatas is given in metas_count
     *
     * Example output:
     *
     * ```
     * [
     *     {"id": 1, "title": "exif metas", "owner": "john", "metas_count": 23},
     *     {"id": 2, "title": "dublin core", "owner": "john", "metas_count": 17}
     * ]
     * ```
     *
     * @returns {Array}
     */
    async metadatasets() {
        return this._rpc("metadatasets", []);
    }


    /**
     * Get a list of available data types
     *
     * Example output:
     *
     * ```
     * [
     *     {"id": 1, "title": "text"},
     *     {"id": 2, "title": "numeric"},
     * ]
     * ```
     *
     * @returns {Array}
     */
    async metadatatypes() {
        return this._rpc("metadatatypes", []);
    }


    /**
     * Move a collection from a parent to another.
     *
     * Will return false in the following cases:
     *
     * - 'child_collection_id' and 'parent_collection_id' are equal
     * - parent collection does not exist
     * - parent collection is a descendant of child collection
     *
     * If 'parent_collection_id' is null, collection is moved to the root.
     *
     * @param {BigInteger} child_collection_id
     * @param {BigInteger} parent_collection_id
     * @returns {Boolean}
     */
    async moveCollection(child_collection_id, parent_collection_id = null) {
        return this._rpc("move_collection", [child_collection_id, parent_collection_id]);
    }


    /**
     * This is a test method to ensure the server-client communication works.
     * Will return "pong [name authenticated of user]"
     *
     * Example output:
     *
     * ```
     * pong john
     * ```
     *
     * @returns {String}
     */
    async ping() {
        return this._rpc("ping", []);
    }


    /**
     * Mark a collection as public
     *
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async publishCollection(collection_id) {
        return this._rpc("publish_collection", [collection_id]);
    }


    /**
     * Remove a meta from a collection given their ids.
     *
     * @param {BigInteger} collection_id
     * @param {BigInteger} meta_value_id
     * @returns {Boolean}
     */
    async removeMetaFromCollection(collection_id, meta_value_id) {
        return this._rpc("remove_meta_from_collection", [collection_id, meta_value_id]);
    }


    /**
     * Remove a meta from a resource given their ids.
     *
     * @param {BigInteger} resource_id
     * @param {BigInteger} meta_value_id
     * @returns {Boolean}
     */
    async removeMetaFromResource(resource_id, meta_value_id) {
        return this._rpc("remove_meta_from_resource", [resource_id, meta_value_id]);
    }


    /**
     * Remove a resource from a collection given ids.
     *
     * @param {BigInteger} resource_id
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async removeResourceFromCollection(resource_id, collection_id) {
        return this._rpc("remove_resource_from_collection", [resource_id, collection_id]);
    }


    /**
     * Remove (delete) a tag based on its uid.
     *
     * Beware: This will remove ALL associations with the tag.
     *
     * @param {String} uid
     * @returns {Boolean}
     */
    async removeTag(uid) {
        return this._rpc("remove_tag", [uid]);
    }


    /**
     * Remove tag from a collection based on tag uid and collection id.
     *
     * @param {String} tag_uid
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async removeTagFromCollection(tag_uid, collection_id) {
        return this._rpc("remove_tag_from_collection", [tag_uid, collection_id]);
    }


    /**
     * Remove tag from a resource based on tag uid and resource id.
     *
     * @param {String} tag_uid
     * @param {BigInteger} resource_id
     * @returns {Boolean}
     */
    async removeTagFromResource(tag_uid, resource_id) {
        return this._rpc("remove_tag_from_resource", [tag_uid, resource_id]);
    }


    /**
     * Rename a collection (ie. change its title).
     *
     * @param {BigInteger} collection_id
     * @param {String} title
     * @returns {Boolean}
     */
    async renameCollection(collection_id, title) {
        return this._rpc("rename_collection", [collection_id, title]);
    }


    /**
     * Rename a metadata (ie. change its title).
     *
     * @param {BigInteger} meta_id
     * @param {String} title
     * @returns {Boolean}
     */
    async renameMeta(meta_id, title) {
        return this._rpc("rename_meta", [meta_id, title]);
    }


    /**
     * Rename a resource (ie. change its title).
     *
     * @param {BigInteger} resource_id
     * @param {String} title
     * @returns {Boolean}
     */
    async renameResource(resource_id, title) {
        return this._rpc("rename_resource", [resource_id, title]);
    }


    /**
     * Replace a file by another using two existing resources.
     *
     * The two resources are expected to be of File type. Then the
     * following operations are performed:
     *
     * - metas from the "ExifTool" set are removed from the destination resource instance
     * - metas from the "ExifTool" set are transfered from the source resource instance to the destination resource instance
     * - the destination resource instance gets the file hash from the source resource instance
     * - the source resource instance is deleted
     * - the destination resource instance is saved
     *
     * Such that all title/metas/tags/collections of the destination resource instance are untouched,
     * excluding exif metas that are transfered from the source.
     *
     * @param {BigInteger} from_resource_id
     * @param {BigInteger} to_resource_id
     * @returns {Boolean}
     */
    async replaceFile(from_resource_id, to_resource_id) {
        return this._rpc("replace_file", [from_resource_id, to_resource_id]);
    }


    /**
     * Get a resource given its id.
     *
     * Example output (file resource):
     *
     * ```
     * {
     *     "id": 1,
     *     "title": "letter",
     *     "original_name": "letter.txt",
     *     "type": "text/plain",
     *     "hash": "0dd93a59aeaccfb6d35b1ff5a49bde1196aa90dfef02892f9aa2ef4087d8738e",
     *     "metas": null,
     *     "urls": [],
     *     "tags": [],
     * }
     * ```
     *
     * @param {BigInteger} resource_id
     * @returns {Object}
     */
    async resource(resource_id) {
        return this._rpc("resource", [resource_id]);
    }


    /**
     * Get all resources from a collection.
     *
     * If 'include_metas' is true, will return the resources metadatas.
     * If 'include_metas' is false, 'metas' will be null.
     *
     * Different resources types may have different object keys. The bare
     * minimum is 'id', 'title' and 'tags'.
     *
     * Example output (file resource):
     *
     * ```
     * [
     *     {
     *         "id": 1,
     *         "title": "letter",
     *         "original_name": "letter.txt",
     *         "type": "text/plain",
     *         "hash": "0dd93a59aeaccfb6d35b1ff5a49bde1196aa90dfef02892f9aa2ef4087d8738e",
     *         "metas": null,
     *         "urls": [],
     *         "tags": [],
     *     }
     * ]
     * ```
     *
     * @param {BigInteger} collection_id
     * @param {Boolean} include_metas
     * @param {BigInteger} limit_from
     * @param {BigInteger} limit_to
     * @returns {Array}
     */
    async resources(collection_id, include_metas = false, limit_from = 0, limit_to = 2000) {
        return this._rpc("resources", [collection_id, include_metas, limit_from, limit_to]);
    }


    /**
     * Set/unset the collection as a OAI-PMH record. The creation date
     * will be used in OAI-PMH requests.
     *
     * @param {BigInteger} collection_id
     * @param {Boolean} is_oai_record
     * @returns {Boolean}
     */
    async setIsOaiRecord(collection_id, is_oai_record = true) {
        return this._rpc("set_is_oai_record", [collection_id, is_oai_record]);
    }


    /**
     * Choose a Resource that is the best representation of a collection.
     * Typical use case: set a miniature for a collection.
     *
     * The Resource does not have to be contained in the collection.
     *
     * Resource id may be set set to None/Null.
     *
     * @param {BigInteger} collection_id
     * @param {BigInteger} resource_id
     * @returns {Boolean}
     */
    async setRepresentativeResource(collection_id, resource_id = null) {
        return this._rpc("set_representative_resource", [collection_id, resource_id]);
    }


    /**
     * Get or create a Tag by uid (unique identifier). 'label' is an optional human-readable name.
     *
     * Example output:
     *
     * ```
     * {
     *     "id": 1,
     *     "uid": "PAINTINGS",
     *     "label": "peintures",
     *     "ark": null,
     * }
     * ```
     *
     * @param {String} uid
     * @param {String} label
     * @param {String} ark
     * @returns {Object}
     */
    async setTag(uid, label = null, ark = null) {
        return this._rpc("set_tag", [uid, label, ark]);
    }


    /**
     * Performs a simple search on resources and collections, based on their titles.
     *
     * Example output:
     *
     * ```
     * {
     *     "collections": [
     *         {
     *         "id": 1,
     *         "title": "photos",
     *         "resources_count": 0,
     *         "children_count": 0,
     *         "descendants_count": 0,
     *         "descendants_resources_count": 0,
     *         "parent": null,
     *         "children": null,
     *         "metas": [],
     *         "public_access": false,
     *         "tags": [],
     *         }
     *     ],
     *     "resources": [
     *         {
     *         "id": 1,
     *         "title": "letter",
     *         "original_name": "letter.txt",
     *         "type": "text/plain",
     *         "hash": "0dd93a59aeaccfb6d35b1ff5a49bde1196aa90dfef02892f9aa2ef4087d8738e",
     *         "metas": null,
     *         "urls": [],
     *         "tags": [],
     *         }
     *     ]
     * }
     * ```
     *
     * @param {String} query
     * @returns {Object}
     */
    async simpleSearch(query) {
        return this._rpc("simple_search", [query]);
    }


    /**
     * Get a list of all supported file type, complete with their mimes.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *     "mime": "image/jpeg",
     *     "extensions": [".jpg", ".jpeg"],
     *     "iiif_support": true,
     *     }
     * ]
     * ```
     *
     * @returns {Array}
     */
    async supportedFileTypes() {
        return this._rpc("supported_file_types", []);
    }


    /**
     * Returns all tags available to the current user.
     *
     * Example output:
     *
     * ```
     * [
     *     {
     *     "id": 1,
     *     "uid": "PAINTINGS",
     *     "label": "peintures",
     *     "ark": null,
     *     },
     *     {
     *     "id": 2,
     *     "uid": "PHOTOS",
     *     "label": "photos",
     *     "ark": null,
     *     }
     * ]
     * ```
     *
     * @returns {Array}
     */
    async tags() {
        return this._rpc("tags", []);
    }


    /**
     * Mark a collection as private
     *
     * @param {BigInteger} collection_id
     * @returns {Boolean}
     */
    async unpublishCollection(collection_id) {
        return this._rpc("unpublish_collection", [collection_id]);
    }


    /**
     * Get information for an upload based on the file hash.
     *
     * Example output:
     *
     * ```
     * {
     *     "status": "not available",
     *     "id": null,
     *     "available_chunks":[]
     * }
     * ```
     *
     * "status" being one of "not available", "available" or "incomplete"
     *
     * @param {String} sha256_hash
     * @returns {Object}
     */
    async uploadInfos(sha256_hash) {
        return this._rpc("upload_infos", [sha256_hash]);
    }
}

export {JamaClient, upload_events}

